#include <EEPROM.h>

/* SERIAL VARIABLES */
char byteRead = 0;
unsigned char retCode = 0xff;

/* CODE PIN VARIABLES */
unsigned char pinOffset = 0; // Offset in EEPROM where the PIN is stored
unsigned char pinSize = 4;
unsigned char pinBuffer[4];
unsigned char userBuffer[4];

/* 
 * Set up:
 * - Open the serial communication port (baudrate 115200)
 * - Set the buffers containing the PIN to 0
 * - Write that the board is set up
 */
void setup() {
  Serial.begin(115200);
  for(unsigned char i=0; i<pinSize; i++){
    pinBuffer[i] = 0;
    userBuffer[i] = 0;
  }
  Serial.print("Set up!\n");
}

/*
 * Verify PIN
 */
unsigned char verifyPin(unsigned char* buf1, unsigned char* buf2, unsigned char size){
  for(unsigned char i=0; i<size; i++){
    if(buf1[i] != buf2[i])
      return 0xaa;
  }
  return 0x55;
}

/*
 * Copy data from EEPROM at the offset to buffer
 */
void rom2buf(unsigned char* buf, unsigned char offset, unsigned char size){
  for(unsigned char i=0; i<size; i++)
    buf[i] = EEPROM.read(offset+i);
}

/*
 * Copy data from buffer to EEPROM at the given offset and clean the buffer
 */
void buf2rom(unsigned char* buf, unsigned char offset, unsigned char size){
  for(unsigned char i=0; i<size; i++){
    EEPROM.write(offset+i, buf[i]);
    buf[i] = 0;
  }
}
 
/*
 * Main loop
 */
void loop() {  
  /* Check if we have some data on the serial port */
  if (Serial.available() > 0) {
    
    byteRead = Serial.read();
    
    switch(byteRead){
      /* Set code PIN to the 4 received values and save it in EEPROM */
      case 0x01:
        Serial.readBytes(pinBuffer, pinSize);
        buf2rom(pinBuffer, pinOffset, pinSize);
        retCode = 0x00;
        break;

      /* Verify PIN sent by user with PIN stored in EEPROM */
      case 0x02:
        Serial.readBytes(userBuffer, pinSize);
        rom2buf(pinBuffer, pinOffset, pinSize);
        retCode = verifyPin(pinBuffer, userBuffer, pinSize);
        break;

      /* By default, just say you don't know the command */
      default:
        retCode = 0xff;
    }
    /* Send the return code */
    Serial.write(retCode);
    
    /* Clean the communication buffer before starting a new loop */
    Serial.flush();
  }
}
