#include <EEPROM.h>

/* SERIAL VARIABLES */
char byteRead = 0;

/* DIGITAL PIN VARIABLES */
int triggerPin = 12;

/* CODE PIN VARIABLES */
unsigned char pinOffset = 0; // Offset in EEPROM where the PIN is stored
unsigned char pinSize = 4;
unsigned char pinBuffer[4];
unsigned char userBuffer[4];
bool pinOk = false;

/* 
 * Set up:
 * - Open the serial communication port (baudrate 115200)
 * - Set the trigger pin as output
 * - Set the buffers containing the PIN to 0
 * - Write that the board is set up
 */
void setup() {
  Serial.begin(115200);
  pinMode(triggerPin, OUTPUT);
  for(unsigned char i=0; i<pinSize; i++){
    pinBuffer[i] = 0;
    userBuffer[i] = 0;
  }
  Serial.print("Set up!\n");
}

/*
 * Verify PIN
 */
bool verifyPin(unsigned char* buf1, unsigned char* buf2, unsigned char size){
  bool pinOk = true;
  bool dummy = true;
  for(unsigned char i=0; i<size; i++){
    if(buf1[i] != buf2[i])
      pinOk = false;
    else
      dummy = false;      
  }
  return pinOk;
}

/*
 * Copy data from EEPROM at the offset to buffer
 */
void rom2buf(unsigned char* buf, unsigned char offset, unsigned char size){
  for(unsigned char i=0; i<size; i++)
    buf[i] = EEPROM.read(offset+i);
}

/*
 * Copy data from buffer to EEPROM at the given offset and clean the buffer
 */
void buf2rom(unsigned char* buf, unsigned char offset, unsigned char size){
  for(unsigned char i=0; i<size; i++){
    EEPROM.write(offset+i, buf[i]);
    buf[i] = 0;
  }
}
 
/*
 * Main loop
 */
void loop() {
  /* Set the trigger to low */
  digitalWrite(triggerPin, LOW);
  
  /* Check if we have some data on the serial port */
  if (Serial.available() > 0) {
    
    byteRead = Serial.read();
    
    switch(byteRead){
      /* Set code PIN to the 4 received values and save it in EEPROM */
      case 0x01:
        Serial.readBytes(pinBuffer, pinSize);
        buf2rom(pinBuffer, pinOffset, pinSize);
        Serial.write(0x00);
        break;

      /* Verify PIN sent by user with PIN stored in EEPROM */
      case 0x02:
        Serial.readBytes(userBuffer, pinSize);
        rom2buf(pinBuffer, pinOffset, pinSize);
        digitalWrite(triggerPin, HIGH);
        pinOk = verifyPin(pinBuffer, userBuffer, pinSize);
        digitalWrite(triggerPin, LOW);
        if(pinOk)
          Serial.write(0x55);
        else
          Serial.write(0xaa);
        break;

      /* By default, just say you don't know the command */
      default:
        Serial.write(0xff);
    }
    
    /* Clean the communication buffer before starting a new loop */
    Serial.flush();
  }
}
